<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

//================================================================================
// QUERYING
//================================================================================

/**
 * Add new query variables in the list of public query variables, 
 * so that we can use it in our custom URLs.
 * https://codex.wordpress.org/Function_Reference/get_query_var
 */
function add_public_query_vars_for_job_opportunities( $qvars ) {
    // Back-end Job Opportunity listings
    $qvars[] = "job_opportunity_meta_query";
    // Front-end Job Opportunity listings
    $myvars = [
        "viewing",
        "keyword_tag",
        "keywords",
        "deadline",
        "job_domains",
        "job_fields",
        "job_profiles",
        "job_provider",
        "job_status",
        "job_type"
    ];
    foreach ( $myvars as $key ) {
        if ( ! array_key_exists( $key, $qvars ) ) {
            $qvars[] = $key;
        }
    }

    return $qvars;
}
add_filter( 'query_vars', 'add_public_query_vars_for_job_opportunities' );

/**
 * Manipulate query before quering 
 * and map 'job_opportunity_meta_query' custom query argument 
 * to the 'meta_query' WordPress's query argument.
 */
function job_opportunity_pre_get_posts( $query ) {

    // if( function_exists('get_current_screen') && get_current_screen()->parent_base == 'edit' && is_admin() && $query->is_main_query() ) {
    //      //$query->query_vars['suppress_filters'] = true;
    //  }

    /**
     * If we're viewing the administration panel, 
     * and we're quering for "job-opportunity" custom post type, 
     * and there's a "job_opportunity_meta_query" custom query argument present, 
     * and its value is not empty.
     */
    if( is_admin() && get_query_var( 'post_type' ) == 'job-opportunity' && ! empty( get_query_var( 'job_opportunity_meta_query' ) ) && ! empty( $query->query_vars['job_opportunity_meta_query'][0]['value'] ) && $query->is_main_query() ) {

        // Map 'job_opportunity_meta_query' to 'meta_query'
        $query->set( 'meta_query', get_query_var( 'job_opportunity_meta_query' ) );
    }

}
add_action( 'pre_get_posts', 'job_opportunity_pre_get_posts' );

/**
 * Filter by custom taxonomy term and/or keyword, if present.
 * Choose from different query arguments depending on
 * the passed terms and keyword.
 * 
 * @return array $query_args An array of arguments to pass to WP_Query()
 */
function getJobOpportunityQueryArguments($job_opportunity_domain_terms = null, $job_opportunity_field_terms = null, $job_opportunity_profile_terms = null, $job_opportunity_provider_terms = null, $job_opportunity_status_terms = null, $job_opportunity_type_terms = null, $job_opportunity_keyword_terms = null, $deadline_date = null, $keywords = null ) {

    // Get all Job Opportunities by default
    $query_args = [
        'post_type' => ['job-opportunity'],
        //'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit'],
        'post_status' => 'publish',
        'orderby'     => 'date',
        'numberposts' => -1,
        //'numberposts' => '5',
        'posts_per_page' => get_option('posts_per_page'),
        'paged' => ( get_query_var( 'paged' ) ) ? absint( get_query_var( 'paged' ) ) : 1,
    ];

    $tax_query = [];

    if ( !empty($job_opportunity_domain_terms) ) {
        $tax_query[] = [
            'taxonomy' => 'job-opportunity-domain',
            'field' => 'term_id',
            'terms' => $job_opportunity_domain_terms,
            'operator' => 'IN',
        ];
    }

    if ( !empty($job_opportunity_field_terms) ) {
        $tax_query[] = [
            'taxonomy' => 'job-opportunity-field',
            'field' => 'term_id',
            'terms' => $job_opportunity_field_terms,
            'operator' => 'IN',
        ];
    }

    if ( !empty($job_opportunity_profile_terms) ) {
        $tax_query[] = [
            'taxonomy' => 'job-opportunity-profile',
            'field' => 'term_id',
            'terms' => $job_opportunity_profile_terms,
            'operator' => 'IN',
        ];
    }

    if ( !empty($job_opportunity_provider_terms) ) {
        $tax_query[] = [
            'taxonomy' => 'job-opportunity-provider',
            'field' => 'term_id',
            'terms' => $job_opportunity_provider_terms,
            'operator' => 'IN',
        ];
    }

    if ( !empty($job_opportunity_status_terms) ) {
        $tax_query[] = [
            'taxonomy' => 'job-opportunity-status',
            'field' => 'term_id',
            'terms' => $job_opportunity_status_terms,
            'operator' => 'IN',
        ];
    }

    if ( !empty($job_opportunity_type_terms) ) {
        $tax_query[] = [
            'taxonomy' => 'job-opportunity-type',
            'field' => 'term_id',
            'terms' => $job_opportunity_type_terms,
            'operator' => 'IN',
        ];
    }

    if ( !empty($job_opportunity_keyword_terms) ) {
        $tax_query[] = [
            'taxonomy' => 'job-opportunity-keyword',
            'field' => 'term_id',
            'terms' => $job_opportunity_keyword_terms,
            'operator' => 'IN',
        ];
    }

    /**
     * Important/Beware: ACF serialized meta data, meta querying with LIKE comparison = slow performance
     * Not all ACF meta is serialized in the database, so manually check the wp_postmeta table for the meta_value.
     * 
     * https://wordpress.stackexchange.com/questions/298156/meta-query-array-in-values-returns-an-error
     * https://wordpress.stackexchange.com/questions/16709/meta-query-with-meta-values-as-serialize-arrays
     * https://stackoverflow.com/questions/26364378/wp-query-when-meta-value-saved-as-serialized-array
     * https://support.advancedcustomfields.com/forums/topic/using-compare-in-operator-for-meta_query/
     */
    $meta_query = [];

    // if ( !empty($example_terms) ) {
    //     $meta_query[] = [
    //         'key' => 'example_key',
    //         'value' => $example_terms,
    //         'compare' => 'IN', # 'IN' for arrays or 'LIKE' for strings
    //     ];
    // }

    if ( !empty($deadline_date) ) {
        $meta_query[] = [
            'key' => 'job_opportunity_deadline',
            'value' => $deadline_date,
            'compare' => '<',
        ];
    }

    if ( !empty($keywords) ) {
        /**
         * Note: Prepending a keyword with a hyphen will exclude posts matching that keyword.
         * E.g., 'pillow -sofa' will return posts containing ‘pillow’ but not ‘sofa’.
         * 
         * https://developer.wordpress.org/reference/classes/wp_query/#search-parameters
         */
        $query_args['s'] = sanitize_text_field($keywords);
    }

    if ( !empty($tax_query) ) {
        if ( count($tax_query) > 1 ) $tax_query = ['relation' => 'AND'] + $tax_query;
        $query_args['tax_query'] = $tax_query;
    }

    if ( !empty($meta_query) ) {
        if ( count($meta_query) > 1 ) $meta_query = ['relation' => 'OR'] + $meta_query;
        $query_args['meta_query'] = $meta_query;
    }

    return $query_args;

}

//================================================================================
// LISTINGS SCREEN (edit.php) CUSTOMIZATION - COLUMNS, FILTERS etc.
//================================================================================

/**
 * Customize views @edit.php?post_type=job-opportunity
 * More @ https://codex.wordpress.org/Plugin_API/Filter_Reference/views_edit-post
 */
function remove_views_from_job_opportunities($views) {  
    unset($views['mine']);
    return $views;
}
add_filter('views_edit-job-opportunity', 'remove_views_from_job_opportunities');

/**
 * Register columns for the back-end listings of the "job-opportunity" custom post type.
 * https://codex.wordpress.org/Plugin_API/Filter_Reference/manage_$post_type_posts_columns
 */
function register_backend_job_opportunity_columns($columns)
{
    $columns['status'] = __('Status', 'erua');

    return $columns;
}
add_filter('manage_job-opportunity_posts_columns', 'register_backend_job_opportunity_columns');

/**
 * Create the content of the custom columns
 * that were configured with register_backend_job_opportunity_columns()
 * http://justintadlock.com/archives/2011/06/27/custom-columns-for-custom-post-types
 * https://wordpress.stackexchange.com/questions/253640/adding-custom-columns-to-custom-post-types
 * https://wpsmackdown.com/easy-filter-posts-wordpress-admin/
 * https://codex.wordpress.org/WordPress_Query_Vars
 * https://wordpress.stackexchange.com/questions/212519/filter-by-custom-field-in-custom-post-type-on-admin-page
 */
function manage_job_opportunity_backend_custom_columns($column, $post_id)
{
    global $post;

    switch ($column) {

        /* If displaying the "status" column. */
        case 'status':

            /* Get the stages for the post. */
            $terms = get_the_terms($post_id, 'job-opportunity-status');

            /* If no terms were found, output a default message. */
            if ( empty($terms) ) {
                echo __('Unknown', 'erua');
            }
            /* If there is a status */
            else {
                $out = array();
                foreach ($terms as $term) {
                    // "Open"
                    if ($term->term_id == 209) {
                        $class_name = 'job-status open-status';
                    }
                    // "Closed"
                    else if ($term->term_id == 210) {
                        $class_name = 'job-status closed-status';
                    }
                    // "Forthcoming"
                    else if ($term->term_id == 211) {
                        $class_name = 'job-status forthcoming-status';
                    }
                    else {
                        $class_name = 'job-status generic-status';
                    }

                    $out[] = sprintf('<a href="%s" class="%s">%s</a>',
                        esc_url(add_query_arg(['post_type' => $post->post_type, 'job-opportunity-status' => $term->slug], 'edit.php')),
                        $class_name,
                        esc_html(sanitize_term_field('name', $term->name, $term->term_id, 'job-opportunity-status', 'display'))
                    );
                }
                echo join(', ', $out);
            }

            break;

        /* Just break out of the switch statement for everything else. */
        default:
            break;
    }
}
add_action('manage_job-opportunity_posts_custom_column', 'manage_job_opportunity_backend_custom_columns', 10, 2);

/**
 * Add filters based on custom taxonomies,
 * for the Job Opportunity listings (@ edit.php).
 * https://generatewp.com/filtering-posts-by-taxonomies-in-the-dashboard/
 */
function create_backend_filters_for_job_opportunities($post_type, $which)
{

    // Apply this only on a specific post type
    if ('job-opportunity' === $post_type) {

        /**
         * A list of taxonomy slugs to filter by
         * Note: Private and draft posts will be displayed 
         * but won't be counted in the taxonomy's terms.
         */
        $taxonomies = [
            'job-opportunity-domain',
            'job-opportunity-profile',
            'job-opportunity-status',
            'job-opportunity-type'
        ];

        foreach ($taxonomies as $taxonomy_slug) {

            // Retrieve taxonomy data
            $taxonomy_obj = get_taxonomy($taxonomy_slug);
            $taxonomy_name = $taxonomy_obj->labels->name;
            $taxonomy_is_hierarchical = $taxonomy_obj->hierarchical;

            if ( $taxonomy_is_hierarchical ) {

                /**
                 * Retrieve parent terms
                 */
                $top_level_terms = get_terms( [
                    'taxonomy'      => $taxonomy_slug,
                    'parent'        => '0',
                    'hide_empty'    => false,
                    'suppress_filters' => false
                ] );

                // Display filter HTML
                echo "<select name='{$taxonomy_slug}' id='{$taxonomy_slug}' class='postform'>";
                echo '<option value="">' . sprintf( esc_html__('All %s', 'erua'), $taxonomy_name ) . '</option>';
                foreach ( $top_level_terms as $top_level_term ) {

                    // View all Job Opportunities with an awaiting nature (pending, waiting for x's reply etc.) of status
                    $query = get_posts(
                        [
                            'post_type' => $post_type,
                            'fields' => 'ids',
                            'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit'],
                            'numberposts' => -1,
                            'tax_query' => [
                                [
                                    'taxonomy' => $taxonomy_slug,
                                    'field' => 'slug',
                                    'terms' => $top_level_term->slug,
                                    'operator' => 'IN'
                                ]
                            ]
                        ]
                    );
                    wp_reset_postdata();
                    $count = count($query);

                    printf(
                        '<option value="%1$s" %2$s>%3$s (%4$s)</option>',
                        $top_level_term->slug,
                        ((isset($_GET[$taxonomy_slug]) && ($_GET[$taxonomy_slug] == $top_level_term->slug)) ? ' selected="selected"' : ''),
                        $top_level_term->name,
                        $count //$top_level_term->count
                    );

                    $top_term_id = $top_level_term->term_id;

                    $second_level_terms = get_terms( array(
                        'taxonomy' => $taxonomy_slug, # you could also use $taxonomy as defined in the first lines
                        'child_of' => $top_term_id,
                        'parent' => $top_term_id, # disable this line to see more child elements (child-child-child-terms)
                        'hide_empty' => false,
                    ) );
                    foreach ($second_level_terms as $second_level_term) {

                        $query = get_posts(
                            [
                                'post_type' => $post_type,
                                'fields' => 'ids',
                                'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit'],
                                'numberposts' => -1,
                                'tax_query' => [
                                    [
                                        'taxonomy' => $taxonomy_slug,
                                        'field' => 'slug',
                                        'terms' => $second_level_term->slug,
                                        'operator' => 'IN'
                                    ]
                                ]
                            ]
                        );
                        wp_reset_postdata();
                        $count = count($query);

                        printf(
                            '<option value="%1$s" %2$s>%3$s (%4$s)</option>',
                            $second_level_term->slug,
                            ((isset($_GET[$taxonomy_slug]) && ($_GET[$taxonomy_slug] == $second_level_term->slug)) ? ' selected="selected"' : ''),
                            '&nbsp;&nbsp;&nbsp;&nbsp;' . $second_level_term->name,
                            $count//$second_level_term->count
                        );
                    }
                }
                echo '</select>';
            }
            else {

                /**
                 * Retrieve parent terms
                 */
                $top_level_terms = get_terms( [
                    'taxonomy'      => $taxonomy_slug,
                    'parent'        => '0',
                    'hide_empty'    => false,
                    'suppress_filters' => false
                ] );

                // Display filter HTML
                echo "<select name='{$taxonomy_slug}' id='{$taxonomy_slug}' class='postform'>";
                echo '<option value="">' . sprintf( esc_html__('Show All %s', 'erua'), $taxonomy_name ) . '</option>';
                foreach ( $top_level_terms as $top_level_term ) {

                    // View all Job Opportunities with an awaiting nature (pending, waiting for x's reply etc.) of status
                    $query = get_posts(
                        [
                            'post_type' => $post_type,
                            'fields' => 'ids',
                            'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit'],
                            'numberposts' => -1,
                            'tax_query' => [
                                [
                                    'taxonomy' => $taxonomy_slug,
                                    'field' => 'slug',
                                    'terms' => $top_level_term->slug,
                                    'operator' => 'IN'
                                ]
                            ]
                        ]
                    );
                    wp_reset_postdata();
                    $count = count($query);

                    printf(
                        '<option value="%1$s" %2$s>%3$s (%4$s)</option>',
                        $top_level_term->slug,
                        ((isset($_GET[$taxonomy_slug]) && ($_GET[$taxonomy_slug] == $top_level_term->slug)) ? ' selected="selected"' : ''),
                        $top_level_term->name,
                        $count //$top_level_term->count
                    );
                }
                echo '</select>';
            }
        }

    }

}
add_action('restrict_manage_posts', 'create_backend_filters_for_job_opportunities', 10, 2);

//================================================================================
// BUBBLE NOTIFICATIONS
//================================================================================

/**
 * Notification "bubble" with pending/open Job Opportunities count
 */
add_action('admin_menu', function () {
    global $menu;
    //$count_posts = wp_count_posts('job-opportunity');
    //$count = $count_posts->pending;

    // View all Job Opportunities with an awaiting nature (pending, waiting for x's reply etc.) of status
    $job_opportunities_query = get_posts(
        [
            'post_type' => 'job-opportunity',
            'fields' => 'ids',
            //'post_status' => 'any', // Published, Pending etc.
            //'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit', 'trash'],
            //'post_status' => ['publish', 'pending', 'draft', 'future', 'private', 'inherit'],
            'post_status' => ['pending', 'draft'],
            'numberposts' => -1,
            // All Job Opportunities that are not solved or closed i.e. pending or open
            // 'tax_query' => [
            //     [
            //         'taxonomy' => 'job-opportunity-status',
            //         'field' => 'slug',
            //         'terms' => ['closed', 'solved'],
            //         'operator' => 'NOT IN',
            //     ],
            // ],
        ]
    );
    if (!empty($job_opportunities_query)) wp_reset_postdata();
    $count = count($job_opportunities_query);
    /* echo '<pre>';
    var_dump($job_opportunities_query);
    echo '</pre>';
    die(); */

    $menu_item = wp_list_filter(
        $menu,
        [2 => 'edit.php?post_type=job-opportunity'] // 2 is the position of an array item which contains URL, it will always be 2!
    );
    if ( !empty($menu_item) && $count > 0 ) {
        $menu_item_position = key($menu_item); // get the array key (position) of the element
        $menu[$menu_item_position][0] .= ' <span class="awaiting-mod" title="' . $count . ' Job Opportunities require actions">' . $count . '</span>';
    }
});
